/*
	Thingy Axon library

	Thingy Axon library is library that enables connectivity to Thingy.IO system.

	Copyright (c) 2017 Thingy.IO
*/

#include "Thingy.h"

namespace Thingy
{

	Axon::Axon() : mqttClient(wifiClient)
	{
		WiFi.disconnect(true);
		WiFi.softAPdisconnect(true);
	}

	// used for setting up an axon - uid/sid taken from settings
	void Axon::setup()
	{
		LOG << F("◤ Setup") << endl;

		// init data
		LOG << F("   Initializing data    : ");
		chipId = String(ESP.getChipId(), DEC);

		LOG << F("✔") << endl;


		LOG << F("   Mounting FileSystem  : ");
		if (SPIFFS.begin())
		{
			LOG << F("✔") << endl;
		}
		else
		{
			LOG << F("✖") << endl;
		}

		// settings
		LOG << F("   Reading settings     : ");
		if (settings_load())
		{
			LOG << F("✔") << endl;
			//LOG << settings << endl;
		}
		else
		{
			LOG << F("✖") << endl;

			LOG << F("   Defaulting settings  : ");
			if (settings_default() && settings_load())
			{
				LOG << F("✔") << endl;
			}
			else
			{
				LOG << F("✖") << endl;
			}
		}


		LOG << F("◣") << endl << endl;
	}

	Axon& Axon::set_wifi_ssid(String ssid)
	{
		this->wifi_ssid = ssid;
		return *this;
	}

	Axon& Axon::set_wifi_pass(String pass)
	{
		this->wifi_pass = pass;
		return *this;
	}

	Axon& Axon::set_uid(String uid)
	{
		this->uid = uid;
		return *this;
	}

	Axon& Axon::set_sid(String sid)
	{
		this->sid = sid;
		return *this;
	}

	bool Axon::begin()
	{
		LOG << F("# ESP") << endl;

		// ESP diagnostics
		LOG << F("   Chip ID              : ") << ESP.getChipId() << endl;
		LOG << F("   Core version         : ") << ESP.getCoreVersion() << endl;
		LOG << F("   SDK version          : ") << ESP.getSdkVersion() << endl;
		LOG << F("   Boot version         : ") << ESP.getBootVersion() << endl;
		LOG << F("   Boot mode            : ") << ESP.getBootMode() << endl;

		LOG << F("   VCC                  : ") << ESP.getVcc() << endl;
		LOG << F("   Free Heap            : ") << ESP.getFreeHeap() << endl;
		LOG << F("   CPU Freq             : ") << ESP.getCpuFreqMHz() << endl;

		LOG << F("   Flash Chip ID        : ") << ESP.getFlashChipId() << endl;
		LOG << F("   Flash Chip real size : ") << ESP.getFlashChipRealSize() << endl;
		LOG << F("   Flash Chip size      : ") << ESP.getFlashChipSize() << endl;
		LOG << F("   Flash Chip speed     : ") << ESP.getFlashChipSpeed() << endl;
		LOG << F("   Flash Chip mode      : ") << ESP.getFlashChipMode() << endl;

		LOG << F("   Sketch size          : ") << ESP.getSketchSize() << endl;
		LOG << F("   Sketch MD5 hash      : ") << ESP.getSketchMD5() << endl;
		LOG << F("   Free sketch space    : ") << ESP.getFreeSketchSpace() << endl;

		LOG << F("   Reset reason         : ") << ESP.getResetReason() << endl;
		LOG << F("   Reset info           : ") << ESP.getResetInfo() << endl;

		LOG << F("#") << endl << endl;

		// initializing system
		hooked = false;
		if (rf_power < 1 || rf_power > 80)
			rf_power = 80;
		system_phy_set_max_tpw(rf_power);

		//
		if(ap_ssid != "")
		{
			// WiFi.softAP(ap_ssid.c_str(), ap_pass.c_str(), WiFi.channel(), 1);
			//WiFi.hostname("Thingy." + chipId);
		}

		//diagnostics
		LOG << F("# WiFi") << endl;
		LOG << F("   WiFi Hostname        : ") << WiFi.hostname() << endl;

		LOG << F("   WiFi MAC address     : ") << WiFi.macAddress() << endl;
		LOG << F("   WiFi SSID            : ") << wifi_ssid << endl;
		LOG << F("   WiFi Password        : ") << wifi_pass << endl;

		if (ap_ssid != "")
		{
			LOG << F("   WiFi AP MAC address  : ") << WiFi.softAPmacAddress() << endl;
			LOG << F("   WiFi AP SSID         : ") << ap_ssid << endl;
			LOG << F("   WiFi AP Password     : ") << ap_pass << endl;
			LOG << F("   WiFi AP RF Power     : ") << rf_power << endl;
		}
		else
		{
			LOG << F("   WiFi AP              : disabled") << endl;
		}
		LOG << F("#") << endl << endl;
		//

		// Over The Air
		String ota_hostname = "Axon."+chipId;
		String ota_password = "CoNnEcToMe";
		ArduinoOTA.setHostname(ota_hostname.c_str());
		ArduinoOTA.setPassword(ota_password.c_str());

		ArduinoOTA.onStart([]() {
			LOG << F("# OTA - Start") << endl;
		});
		ArduinoOTA.onEnd([]() {
			LOG << endl << F("# OTA - End") << endl << endl;
		});
		ArduinoOTA.onProgress([](unsigned int progress, unsigned int total) {
			LOG << F("   ") << (0.01f*round(10000.0 * progress / total)) << F(" %") << endl;
		});
		ArduinoOTA.onError([](ota_error_t error) {
			LOG << endl << F("# OTA - ERROR - ") << error;
			if (error == OTA_AUTH_ERROR) LOG << F("Auth Failed") << endl;
			else if (error == OTA_BEGIN_ERROR) LOG << F("Begin Failed") << endl;
			else if (error == OTA_CONNECT_ERROR) LOG << F("Connect Failed") << endl;
			else if (error == OTA_RECEIVE_ERROR) LOG << F("Receive Failed") << endl;
			else if (error == OTA_END_ERROR) LOG << F("End Failed") << endl;
		});
		//

		// connecting to WiFi
		if (!wifi_connect())
		{
		  settings_web_portal();
			return false;
		}

		// Beggining Over The Air update listener
		ArduinoOTA.begin();
		LOG << F("# OTA (Over the air) update") << endl;
		//LOG << F("   Listener started : ") << (ota ? F("OK") : F("ERROR")) << endl;
		LOG << F("   Listening IP     : ") << WiFi.localIP() << endl;
		LOG << F("   Update password  : ") << ota_password << endl;
		LOG << F("#") << endl << endl;

		// diagnostics
		LOG << F("# Thingy") << endl;
		LOG << F("   Axon API endpoint    : ") << thingy_server << " : " << thingy_port << endl;
		LOG << F("   Axon Unique ID       : ") << uid << endl;
		LOG << F("   Axon Security ID     : ") << sid << endl;
		LOG << F("#") << endl << endl;

		if (thingy_server == NULL || thingy_server == "" ||
		    uid == NULL || uid == "" ||
				sid == NULL || sid == "")
		{
			settings_web_portal();
			return false;
		}

		int head_retries = 3;
		while (head_retries-->0)
		{
			if (hooked = head()) break;
			yield();
		}
		if (head_retries<0)
		{
			settings_web_portal();
			return false;
		}

		LOG << F("# Thingy") << endl;
		LOG << F("   Brain Unique ID      : ") << brain << endl;
	  LOG << F("   Axon Unique ID       : ") << uid << endl;
		LOG << F("   Axon External ID     : ") << eid << endl;
		LOG << F("   Axon Name            : ") << name << endl;

		LOG << F("#") << endl;
		for (int ai=0; ai < streams_len; ai++)
		{

			LOG << F("   Stream Unique ID     : ") << (streams[ai].uid) << endl;
			LOG << F("   Stream Name          : ") << (streams[ai].name) << endl;
			LOG << F("   Stream type          : ") << (Helper::parse(streams[ai].type)) << endl;
			LOG << F("   Stream ACL           : ") << (Helper::parse(streams[ai].acl)) << endl;
			LOG << F("#") << endl;
		}
		LOG << endl;

		// MQTT
		mqttClient.set_server(mqtt_server, mqtt_port);
		mqttClient.set_callback([this] (const MQTT::Publish& pub) {
			this->got(pub);
		});
		//mqttClient.rfPWR = rf_power;
		mqtt_client_id = "Thingy." + chipId;

		//diagnostics
		LOG << F("# MQTT") << endl;
		LOG << F("   MQTT Server          : ") << mqtt_server << " : " << mqtt_port << endl;
		LOG << F("   MQTT Username        : ") << uid << endl;
		LOG << F("   MQTT Password        : ") << sid << endl;
		LOG << F("   MQTT Topic root      : ") << mqtt_topic << endl;
		LOG << F("   MQTT Client ID       : ") << mqtt_client_id << endl;
		LOG << F("#") << endl << endl;

		if (!mqtt_connect())
		{
		  settings_web_portal();
			return false;
		}
		return true;
	}

	bool Axon::head()
	{
		LOG << F("* Thingy") << endl;

		LOG << F("   Connecting           : ");
		if (wifiSecureClient.connect(thingy_server.c_str(), thingy_port))
		{
			LOG << F("OK") << endl;
		}
		else
		{
			LOG << F("ERROR") << endl;
			LOG << F("*") << endl << endl;
			return false;
		}

		LOG << F("   Checking certificate : ");
		if (wifiSecureClient.verify(thingy_finger.c_str(), thingy_server.c_str()))
		{
			LOG << F("Verified") << endl;
		}
		else
		{
			LOG << F("Unverified") << endl;
			LOG << F("*") << endl << endl;
			//return false;
		}

		LOG << F("   Requesting           : ");

		String data = "{\"last_seen\":\"2000-01-01\",\"EiD\":\""+chipId+"\",\"ip\":\""+wifi_ip+"\",\"data\":\""+ESP.getSketchMD5()+"\"}";

		// existing axon, retreiave master data
		wifiSecureClient << F("PATCH /axons/") << uid << F("?fields=UiD,EiD,name,ip,brain,settings&expand=streams HTTP/1.1") << endl;
		wifiSecureClient << F("Host: ") << thingy_server << endl;
		wifiSecureClient << F("User-Agent: Thingy.") << chipId << endl;
		wifiSecureClient << F("Authorization: Basic ") << base64::encode(sid+":") << endl;
		//wifiSecureClient << F("Authorization: Bearer ") << base64::encode(sid) << endl;
		wifiSecureClient << F("Content-Type: application/json") << endl;
		wifiSecureClient << F("Content-Length: ") << data.length() << endl;
		wifiSecureClient << F("Connection: close") << endl;
		wifiSecureClient << endl;
		wifiSecureClient << data;

		LOG << F("OK") << endl;

		LOG << F("   Receiving            : ");

		{
			int timeout = millis() + 15000;
			while (wifiSecureClient.available() == 0) {
				if (timeout - millis() < 0) {
					LOG << F("ERROR (timeout)") << endl;
					LOG << F("*") << endl << endl;
					wifiSecureClient.stop();
					return false;
				}
			}
		}

		//LOG << endl << F("===== headers =====") << endl;
		while (wifiSecureClient.available())
		{
			String line = wifiSecureClient.readStringUntil('\n');

			if (line == "\r")
			{
				break;
			}
			else
			{
				//LOG << line << endl;
			}
		}
		//LOG << F("===================") << endl << endl;

		{
			int timeout = millis() + 15000;
			while (wifiSecureClient.available() == 0) {
				if (timeout - millis() < 0) {
					LOG << F("ERROR (timeout)") << endl;
					LOG << F("*") << endl << endl;
					wifiSecureClient.stop();
					return false;
				}
			}
		}

		String response;
		if (wifiSecureClient.available())
		{
			response = wifiSecureClient.readStringUntil('\r');
			LOG << F("OK") << endl;
		}
		else
		{
			LOG << F("ERROR") << endl;
			LOG << F("*") << endl << endl;
			return false;
		}
		yield();

		//LOG << F("===== response =====") << endl << response << endl << F("====================") << endl << endl;

		LOG << F("   Parsing response     : ");
		DynamicJsonBuffer jsonBuffer;
		JsonObject& axon = jsonBuffer.parseObject(response);
		if (axon.success())
		{
			LOG << F("OK") << endl;
		}
		else
		{
			LOG << F("ERROR (Parsing response)") << endl;
			LOG << F("*") << endl << endl;
			return false;
		}

		// basic properties
		if (axon.containsKey("UiD") &&
				axon.containsKey("EiD") &&
				axon.containsKey("name") &&
				axon.containsKey("brain"))
		{
			uid = axon.get<String>("UiD");
			eid = axon.get<String>("EiD");
			name = axon.get<String>("name");
			brain = axon.get<String>("brain");
		}
		else
		{
			LOG << F("ERROR (Extracting response)") << endl;
			LOG << F("*") << endl << endl;
			return false;
		}

		mqtt_topic = "brain/" + brain + "/axon/" + uid;

		// settings update

		DynamicJsonBuffer settingsBuffer;
		JsonObject& settingsObject = settingsBuffer.parseObject(axon.get<String>("settings"));

		String _settings = "";
		settingsObject.printTo(_settings);

		if (settingsObject.success() && settingsObject.containsKey("timestamp") &&
			settings != _settings)
		{
			LOG << F("S1") << endl;

			File settingsFile = SPIFFS.open("/settings.json", "w");
			if (settingsFile)
			{
				LOG << F("S3") << endl;
				settingsObject.printTo(settingsFile);
				settingsFile.close();

				LOG << F("### Restarting with new settings") << endl;

				ESP.restart();
			}
			else
			{
				LOG << F("S4") << endl;
			}
		}
		else
		{
			//LOG << F("S2") << endl;
		}

		// streams
		streams_len = 0;

		if (axon.containsKey("streams"))
		{
			JsonArray& streamsArray = axon["streams"];
			for(JsonArray::iterator it=streamsArray.begin(); it!=streamsArray.end(); ++it,streams_len++) { }

		  streams = new Stream[streams_len];
			int streams_i = 0;

			for(JsonArray::iterator it=streamsArray.begin(); it!=streamsArray.end(); ++it,streams_i++)
			{
				JsonObject& streamObject = *it;
				//streamObject.prettyPrintTo(LOG);
				streams[streams_i].axon = this;
				streams[streams_i].uid = streamObject.get<String>("UiD");
				streams[streams_i].name = streamObject.get<String>("name");
				if (streamObject.get<String>("type") == "Dendrite")
					streams[streams_i].type = StreamType::Dendrite;
				if (streamObject.get<String>("type") == "Terminal")
					streams[streams_i].type = StreamType::Terminal;
				if (streamObject.get<String>("acl") == "system")
					streams[streams_i].acl = StreamAccessLevel::System;
				if (streamObject.get<String>("acl") == "private")
					streams[streams_i].acl = StreamAccessLevel::Private;
				if (streamObject.get<String>("acl") == "public")
					streams[streams_i].acl = StreamAccessLevel::Public;
				streams[streams_i].mqtt_topic = mqtt_topic+"/stream/" + streams[streams_i].uid;
			}
		}

		wifiSecureClient.stop();

		LOG << F("*") << endl << endl;
		return true;
	}

	bool Axon::tail(bool full = false)
	{
		bool ret = true;

		//LOG << F("> TAIL axon   ") << uid << F(" @ ") << ticks() << F(" - ");

		if (full)
		{
			ret &= mqttClient.publish(MQTT::Publish(mqtt_topic+"/uid", uid).set_qos(1).set_retain(true));
			ret &= mqttClient.publish(MQTT::Publish(mqtt_topic+"/eid", eid).set_qos(1).set_retain(true));
			ret &= mqttClient.publish(MQTT::Publish(mqtt_topic+"/name", name).set_qos(1).set_retain(true));
			ret &= mqttClient.publish(MQTT::Publish(mqtt_topic+"/settings", settings).set_qos(1).set_retain(true));
			ret &= mqttClient.publish(MQTT::Publish(mqtt_topic+"/ip", wifi_ip).set_qos(1).set_retain(true));
			ret &= mqttClient.publish(MQTT::Publish(mqtt_topic+"/sketch_size", String(ESP.getSketchSize(), DEC)).set_qos(1).set_retain(true));
			ret &= mqttClient.publish(MQTT::Publish(mqtt_topic+"/sketch_free",  String(ESP.getFreeSketchSpace(), DEC)).set_qos(1).set_retain(true));
			ret &= mqttClient.publish(MQTT::Publish(mqtt_topic+"/sketch_md5", ESP.getSketchMD5()).set_qos(1).set_retain(true));
			ret &= mqttClient.publish(MQTT::Publish(mqtt_topic+"/reset_reason", ESP.getResetReason()).set_qos(1).set_retain(true));
			ret &= mqttClient.publish(MQTT::Publish(mqtt_topic+"/reset_info", ESP.getResetInfo()).set_qos(1).set_retain(true));
		}
		//diagnostics
		ret &= mqttClient.publish(MQTT::Publish(mqtt_topic+"/ticks", String(_ticks)).set_qos(1));
		ret &= mqttClient.publish(MQTT::Publish(mqtt_topic+"/rssi", String(WiFi.RSSI(), DEC)).set_qos(1));
		ret &= mqttClient.publish(MQTT::Publish(mqtt_topic+"/freeheap", String(ESP.getFreeHeap(), DEC)).set_qos(1));

		yield();

		for (int i=0; i < streams_len; i++)
			ret &= streams[i].tail(full);

		//LOG << (ret ? F("OK") : F("ERROR")) << endl;
		if (!ret) LOG << F("> TAIL axon   ") << uid << F(" @ ") << ticks() << F(" - ERROR") << endl;

		return ret;
	}

	bool Axon::connected()
	{
		return (WiFi.status() == WL_CONNECTED) && mqttClient.connected() && hooked;
	}

	// wifi
	void Axon::wifi_setup(String ssid, String password)
	{
		wifi_ssid = ssid;
		wifi_pass = password;
	}

	bool Axon::wifi_connect()
	{
		bool connected = false;

		LOG << F("* WiFi") << endl;

		if (WiFi.status() == WL_CONNECTED)
		{
			IPAddress ip = WiFi.localIP();
			wifi_ip = String(ip[0]) + '.' + String(ip[1]) + '.' + String(ip[2]) + '.' + String(ip[3]);

			LOG << F("   Connecting           : ");
			LOG << F("OK (Already connected") << endl;
			connected = true;
		}
		else
		{
			/// ***
			WiFi.disconnect(true);
			delay(500);
			/// ***

			int wifi_retries = 5;
			while (WiFi.status() != WL_CONNECTED && wifi_retries-->0)
			{
			  LOG << F("   Connecting       [") << (5-wifi_retries) << F("] : ");
				WiFi.begin(wifi_ssid.c_str(), wifi_pass.c_str());
				if (WiFi.waitForConnectResult() == WL_CONNECTED)
				{
					IPAddress ip = WiFi.localIP();
					wifi_ip = String(ip[0]) + '.' + String(ip[1]) + '.' + String(ip[2]) + '.' + String(ip[3]);

					LOG << F("OK") << endl;
					LOG << F("   IP address           : ") << wifi_ip << endl;
					LOG << F("   Subnet mask          : ") << WiFi.subnetMask() << endl;
					LOG << F("   Gateway IP           : ") << WiFi.gatewayIP() << endl;
					connected = true;
				}
				else
				{
					LOG << F("ERROR") << endl;
					connected = false;
				}
				yield();
			}
		}
		LOG << F("*") << endl << endl;

		return connected;
	}

	bool Axon::wifi_connected()
	{
		return (WiFi.status() == WL_CONNECTED);
	}

	bool Axon::mqtt_connect()
	{
		bool connected = false;

		LOG << F("* MQTT") << endl;

		if (mqttClient.connected())
		{
			LOG << F("   Connecting           : ");
			LOG << F("OK (Already connected") << endl;
			connected = true;
		}
		else
		{
			/// ***
			mqttClient.disconnect();
			/// ***

			MQTT::Connect mqttConnect =
			MQTT::Connect(mqtt_client_id)
				.set_auth(uid, sid)
				.set_clean_session(true)
				.set_keepalive(30)
				.set_will(mqtt_topic + "/will", String(++drops, DEC), 2, true);

			int mqtt_retries = 5;
			while (!mqttClient.connected() && mqtt_retries-->0)
			{
				LOG << F("   Connecting       [") << (5-mqtt_retries) << F("] : ");
				if (mqttClient.connect(mqttConnect))
				{
					yield();

					LOG << F("OK") << endl;
					LOG << F("   MQTT Client ID       : ") << mqtt_client_id << endl;
					tail(true);

					// make subscription
					LOG << F("   Subscribing") << endl;

					bool subscribed = true;
					subscribed &= mqttClient.subscribe(mqtt_topic + "/req");
					LOG << F("      ") << (subscribed ? F("+ ") : F("- ")) << (mqtt_topic + "/req") << endl;
					yield();

					subscribed &= mqttClient.subscribe(mqtt_topic + "/settings");
					LOG << F("      ") << (subscribed ? F("+ ") : F("- ")) << (mqtt_topic + "/settings") << endl;
					yield();

					for (int streams_i=0; streams_i < streams_len; streams_i++)
					{
						if (streams[streams_i].type == StreamType::Dendrite)
						{
							subscribed &= mqttClient.subscribe(streams[streams_i].mqtt_topic);
							yield();
							LOG << F("      ") << (subscribed ? F("+ ") : F("- ")) << streams[streams_i].mqtt_topic << endl;
						}
					}

					//
					/*
					MQTT::Subscribe subscriptions = MQTT::Subscribe()
						.add_topic(mqtt_topic);
					  //.add_topic(String(mqtt_topic + "/settings"));

						LOG << "mqtt_connect 2" << endl;
					for (int streams_i=0; streams_i < streams_len; streams_i++)
					{
						if (streams[streams_i].type == StreamType::Dendrite)
						{
							//subscriptions = subscriptions.add_topic(streams[streams_i].mqtt_topic);
						}
					}
					LOG << F("   Subscribing          : ");
					bool subscribed = mqttClient.subscribe(subscriptions);
					LOG << (subscribed ? F("✔ ") : F("✖ ")) << endl;
					*/

					connected = true;
				}
				else
				{
					LOG << F("ERROR") << endl;
					//LOG.print(mqttClient.state());
					connected = false;
				}
			}
		}

		LOG << F("*") << endl << endl;

		return connected;
	}

	void Axon::mqtt_disconnect()
	{
		mqttClient.disconnect();
		yield();
	}

	bool Axon::mqtt_loop()
	{
		return mqttClient.loop();
	}

	bool Axon::mqtt_connected()
	{
		return mqttClient.connected();
	}

	void Axon::got(const MQTT::Publish& pub)
	{
		//LOG << F("> MQTT Published - ") << pub.topic() << " « " << pub.payload_string() << endl;
		for (int streams_i=0; streams_i<streams_len; streams_i++)
		{
				if ((pub.topic() == streams[streams_i].mqtt_topic) &&
			    (streams[streams_i].type == StreamType::Dendrite))
			{
				streams[streams_i].got(pub);
				return;
			}
		}

		/*
		// settings
		if (pub.topic() == mqtt_topic + "settings")
		{
			LOG << F("* MQTT - Settings update from MQTT") << endl;
			String newSettings = pub.payload_string();
			LOG << newSettings << endl;
			StaticJsonBuffer<500> settingsBuffer;
			JsonObject& settingsObject = settingsBuffer.parseObject(newSettings);

			if (settingsObject.success())
			{
				LOG << F("1");
				String _settings;
				settingsObject.printTo(_settings);

				if (settings == _settings)
				{
					LOG << F("3");
					File settingsFile = SPIFFS.open("/settings.json", "w");
					if (settingsFile)
					{
						LOG << F("5");
						settingsObject.printTo(settingsFile);
						settingsFile.close();

						LOG << F("* MQTT - Restarting with new settings") << endl;

						ESP.restart();
					}
					else
					{
						LOG << F("6") << endl;
					}
				}
				else
				{
					LOG << F("4") << endl;
				}
			}
			else
			{
				LOG << F("2") << endl;;
			}
		}
		*/

		// command processing (Steva)
		/*
		if (pub.topic() == mqtt_topic + "/req")
		{

			String command;
			String response = "";
			char *tokens[10];
			//using payload_string(), as payload(); has trailing garbage
			String commandSetString = pub.payload_string();
			//moving string to *char, to use strtok, as arduino string functions are shit
			char commandSet[64];
			commandSetString.toCharArray(commandSet, 64);
			tokens[0] = strtok(commandSet, "\n\r");
			int i = 0;
			while ((tokens[i] != NULL)&&(i < 9)){
				i++;
				tokens[i] = strtok(NULL, "\n\r");
			}
			for (int j = 0; j < i; j++){
				command = String(tokens[j]);
				response = processCommand(command);
				//response += "\r\n";
			}

			MQTT::Publish newpub(mqtt_topic + "/res", response);
			mqttClient.publish(newpub.set_qos(2));

			return;
		}
		*/

		LOG << F("> GOT axon ") << uid;
		if (handle)
		{
			LOG << F(" & handler invoked") << endl;
			String topic = static_cast<String>(pub.topic());
			String message = static_cast<String>(pub.payload_string());

			handle(*this, topic, message, pub);
			yield();
		}
		else
			LOG << endl;
	}

	unsigned long m01 = 0;
	unsigned long m05 = 0;
	unsigned long m60 = 0;

	void Axon::loop()
	{
		// hook and grab settings
		/*
		axon.setupLoop();
		if (axon.setupMode) return; //if setup mode, don't do anything and wait for restart
		*/

		// handle mqtt connection
		if (!mqtt_loop())
		{
			yield();
			LOG << F("# MQTT - not looped - disconnecting ... @ ") << ticks() << endl;
			mqtt_disconnect();
			yield();
			LOG << F("# MQTT - not looped - ... disconnected and reconnecting ... @ ") << ticks() << endl;
			mqtt_connect();
			yield();
			LOG << F("# MQTT - not looped - ... connected @ ") << ticks() << endl;
		}

		// handle OTA
		ArduinoOTA.handle();
		yield();

		// every 1 second
		if (m01 != 0 && (millis() - m01) < 1000) return;
		m01 = millis();
		_ticks++;
		//

		// every 15 seconds
		if (m05 != 0 && (millis() - m05) < 15000) return;
		m05 = millis();
		//LOG << F("# [") << (connected() ? F("+"):F("-")) << F("] 15s @ ") << ticks() << endl;
		//

		// WiFi (re)connection
		if (!wifi_connected())
		{
			bool reconnected = wifi_connect();
			if (!reconnected)
			{
				restart();
				return;
			}
		}

		// MQTT (re)connection
		if (wifi_connected() && thingy_connected() && !mqtt_connected())
		{
			mqtt_connect();
			yield();
		}

		//if (connected()) tail();

		// every minute
		if (m60 != 0 && (millis() - m60) < 60000) return;
		m60 = millis();
		LOG << F("# [") << (connected() ? F("+"):F("-")) << F("] 1m @ ") << ticks() << endl;
		//

		if (connected()) tail();
	}

	void Axon::restart()
	{
		ESP.restart();
	}

	Thingy::Stream& Axon::operator[](String nameOrUid)
	{
		for (int streams_i=0; streams_i<streams_len; streams_i++)
		{
				if (nameOrUid == streams[streams_i].name ||
				    nameOrUid == streams[streams_i].uid)
					return streams[streams_i];
		}
		//throw std::out_of_range("Stream not found");
	}

	void Axon::scan_wifi_networks(String& table)
	{
		int n = WiFi.scanNetworks(false, true);
		if (n > 30)
			n = 30; //limit maximum wifi send to 30. this should fit in 5000 bytes reserved

		StaticJsonBuffer<1000> jsonBuffer;
		JsonObject& root = jsonBuffer.createObject();
		JsonArray& aps = root.createNestedArray("aps");

		for (int i = 0; i < n; ++i)
		{
			JsonObject& ap = aps.createNestedObject();
			ap["ssid"] = WiFi.SSID(i);
			ap["mac"] = WiFi.BSSIDstr(i);
			//ap["channel"] = WiFi.channel(i);
			//ap["hidden"] = WiFi.isHidden(i);

			/*
			switch (WiFi.encryptionType(i))
			{
				case ENC_TYPE_WEP:
					ap["encryption"] = "WEP";
					break;
				case ENC_TYPE_TKIP:
					ap["encryption"] =  "WPA";
					break;
				case ENC_TYPE_CCMP:
					ap["encryption"] =  "WPA2";
					break;
				case ENC_TYPE_NONE:
					ap["encryption"] =  "None";
					break;
				case ENC_TYPE_AUTO:
					ap["encryption"] =  "Auto";
					break;
			}
			*/
			ap["rssi"] = WiFi.RSSI(i);
		}

		root.prettyPrintTo(table);
	}

	// Web Config
	void Axon::settings_web_portal()
	{
		int timeout = 188;

		LOG << F("* Settings Web Portal") << endl;

		WiFiManagerParameter custom_api("<b>API settings</b>");
		WiFiManagerParameter custom_api_server("Hostname", "Thingy API endpoint", "api.thingy.io", 32, " readonly");
		WiFiManagerParameter custom_mqtt("<b>MQTT settings</b>");
		WiFiManagerParameter custom_mqtt_server("MQTT", "Thingy MQTT endpoint", mqtt_server.c_str(), 32);
			//WiFiManagerParameter custom_mqtt_port("port", "Thingy MQTT port", axon.mqtt_por, 5);
		WiFiManagerParameter custom_thingy("<b>Authentication settings</b>");
		WiFiManagerParameter custom_thingy_uid("UiD", "Unique ID", uid.c_str(), 8);
		WiFiManagerParameter custom_thingy_sid("SiD", "Security ID", sid.c_str(), 32, " type='password'");

		WiFiManager wifiManager;
		wifiManager.setDebugOutput(false);
		wifiManager.setMinimumSignalQuality(35);
		wifiManager.setConfigPortalTimeout(timeout);

		wifiManager.addParameter(&custom_thingy);
		wifiManager.addParameter(&custom_thingy_uid);
		wifiManager.addParameter(&custom_thingy_sid);
		wifiManager.addParameter(&custom_api);
		wifiManager.addParameter(&custom_api_server);
		wifiManager.addParameter(&custom_mqtt);
		wifiManager.addParameter(&custom_mqtt_server);

		LOG << F("   Timeout          : ") << timeout << F("s") << endl;
		LOG << F("   Starting portal  : ");

		wifiManager.setCustomHeadElement("<style>.h {border: 2px solid #1FA3EC;border-radius: 0.3rem;border-width: 4px;text-align: center;line-height: 2.4rem;}</style>");
		// in line 365: page += F("<div class='h'>Thingy<br><img src='data:image/png;base64,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' /><br>Axon</div><br>");
		String name ="Thingy."+String(ESP.getChipId());
		if (wifiManager.startConfigPortal(name.c_str()))
		{
			if (wifi_ssid == WiFi.SSID() && wifi_pass == WiFi.psk() &&
		      mqtt_server == String(custom_mqtt_server.getValue()) &&
					uid == String(custom_thingy_uid.getValue()) &&
					sid == String(custom_thingy_sid.getValue()))
			{
				LOG << F("No chanage") << endl;
				LOG << F("*") << endl << endl;
				return;
			}
			else
				LOG << F("OK") << endl;
		}
		else
		{
			LOG << F("Expired or Canceled") << endl;
			LOG << F("   Restarting device") << endl;
			restart();
			LOG << F("*") << endl << endl;
			return;
		}

		wifi_ssid = WiFi.SSID();
		wifi_pass = WiFi.psk();
		//axon.mqtt_port = String(custom_api_server.getValue());
		mqtt_server = String(custom_mqtt_server.getValue());
		uid = String(custom_thingy_uid.getValue());
		sid = String(custom_thingy_sid.getValue());

		LOG << F("   Saving settings   : ");
		if (settings_save())
			LOG << F("OK") << endl;
		else
			LOG << F("ERROR") << endl;

		LOG << F("   Restarting device") << endl;
		restart();
		LOG << F("*") << endl << endl;
	}

/*
	// auto-hook up
	void Axon::setupLoop()
	{
		if(setupMode)
		{
			if (WiFi.status() != WL_CONNECTED)
			{
				LOG << F(" Looking for master node: ");
				WiFi.begin("default.ssid", "default.pass");
				if (WiFi.waitForConnectResult() != WL_CONNECTED)
				{
					LOG << F("ERROR (Not found)") << endl;
				}
				else
				{
					LOG << F("OK") << endl;
				}
			}
			else
			{
				WiFiClient client;
				IPAddress server(192,168,4,1); //default ip address of esp
				LOG << F(" Connecting to master   : ");

				if (client.connect(server, 80))
				{
					LOG << F("OK") << endl;

					LOG << F(" Synchronizing          : ");
					client.println("access.password");

					String inputLine;
					do
					{
						inputLine = client.readStringUntil('\r');
						processCommand(inputLine);
					}
					while (inputLine != "");

					LOG << F("OK") << endl;

					LOG << F(" Exiting               : ");
					setupMode = false;
					LOG << F("OK") << endl;

					LOG << F("*") << endl;
				}
				else{
					LOG << F("ERROR (Server not found)") << endl;
				}
			}
		}
		else
		{
			if(!digitalRead(0))
			{
				if (setupCounter == 0)
				{
					setupCounter = millis();
				}
				else
				{
					if (millis() - setupCounter > 3000)
					{
						LOG << F("* SETUP MODE") << endl;
						LOG << F(" Entering               : ");

						WiFi.softAPdisconnect(true);
						WiFi.disconnect(true);
						setupMode = true;
						LOG << F("OK") << endl;
					}
				}
			}
			else
			{
				setupCounter = 0;
			}
		}
	}
	*/
	//

	// settings persistance
	bool Axon::settings_load()
	{
		File settingsFile = SPIFFS.open("/settings.json", "r");
		if (!settingsFile)
		{
			LOG << F("Failed to open settings file  ");
			return false;
		}

		size_t size = settingsFile.size();
		if (size > 500)
		{
			LOG << F("Settings file size is too large ");
			return false;
		}

		std::unique_ptr<char[]> buf(new char[size]);
		settingsFile.readBytes(buf.get(), size);

		StaticJsonBuffer<500> settingsBuffer;
		JsonObject& settingsObject = settingsBuffer.parseObject(buf.get());

		settingsFile.close();

		if (!settingsObject.success())
		{
			LOG << F("Failed to parse settings file ");
			return false;
		}

		if (!settingsObject.containsKey("timestamp"))
		{
			LOG << F("Failed to parse settings file ");
			return false;
		}

		settings = "";
		settingsObject.printTo(settings);

		wifi_ssid     = settingsObject.get<String>("wifi.ssid");
		wifi_pass     = settingsObject.get<String>("wifi.pass");

		ap_ssid       = settingsObject.get<String>("ap.ssid");
		ap_pass       = settingsObject.get<String>("ap.pass");

		thingy_server = settingsObject.get<String>("thingy.server");
		thingy_port   = settingsObject.get<int>("thingy.port");
		thingy_finger = settingsObject.get<String>("thingy.finger");

		uid      = settingsObject.get<String>("axon.uid");
		sid      = settingsObject.get<String>("axon.sid");

		mqtt_server   = settingsObject.get<String>("mqtt.server");
		mqtt_port     = settingsObject.get<int>("mqtt.port");

		rf_power      = settingsObject.get<int>("rf.power");

		settings_timestamp = settingsObject.get<unsigned long>("timestamp");

		return true;
	}

	bool Axon::settings_save()
	{
		StaticJsonBuffer<500> settingsBuffer;
		JsonObject& settingsObject = settingsBuffer.createObject();

		settingsObject["wifi.ssid"]       = wifi_ssid;
		settingsObject["wifi.pass"]       = wifi_pass;

		settingsObject["ap.ssid"]         = ap_ssid;
		settingsObject["ap.pass"]         = ap_pass;

		settingsObject["thingy.server"]   =  thingy_server;
		settingsObject["thingy.port"]     =  thingy_port;
		settingsObject["thingy.finger"]   =  thingy_finger;

		settingsObject["axon.uid"]        = uid;
		settingsObject["axon.sid"]        = sid;

		settingsObject["mqtt.server"]     = mqtt_server;
		settingsObject["mqtt.port"]       = mqtt_port;

		settingsObject["rf.power"]        = rf_power;

		settingsObject["timestamp"]       = _ticks;

		File settingsFile = SPIFFS.open("/settings.json", "w");
		if (!settingsFile)
		{
			LOG << F("Failed to open config file for writing ");
			return false;
		}

		settingsObject.printTo(settingsFile);

		settings = "";
		settingsObject.printTo(settings);
		if (DEBUG) settingsObject.printTo(CONS);

		settings_timestamp = _ticks;

		settingsFile.close();

		return true;
	}

	bool Axon::settings_default()
	{
		StaticJsonBuffer<500> settingsBuffer;
		JsonObject& settingsObject = settingsBuffer.createObject();

		settingsObject["wifi.ssid"]       = String("Thingy.SSID");
		settingsObject["wifi.pass"]       = String("Thingy.PASS");

		settingsObject["ap.ssid"]         = String("Thingy."+chipId);
		settingsObject["ap.pass"]         = String("Thingy."+chipId);

		settingsObject["thingy.server"]   = String("api.thingy.io");
		settingsObject["thingy.port"]     = 443;
		settingsObject["thingy.finger"]   = String("B992CA264A39A0E2829E7830498187EE519038FD");

		settingsObject["axon.uid"]        = String(sha1(String(ESP.getChipId(), DEC))).substring(0,8);
		settingsObject["axon.sid"]        = String("");

		settingsObject["mqtt.server"]     = String("mqtt.thingy.io");
		settingsObject["mqtt.port"]       = 8883;

		settingsObject["rf.power"]        = 80;

		settingsObject["timestamp"]       = 0;

		File settingsFile = SPIFFS.open("/settings.json", "w");
		if (!settingsFile)
		{
			LOG << F("Failed to open settings file for writing ");
			return false;
		}

		settingsObject.printTo(settingsFile);
		settings = "";
		settingsObject.printTo(settings);
		settings_timestamp = 0;

		settingsFile.close();

		return true;
	}

	bool Axon::settings_remove()
	{
		return SPIFFS.exists("/settings.json") && SPIFFS.remove("/settings.json");
	}

/*
	String Axon::processCommand(String command)
	{
		command.replace("\n", "");
		command.replace("\r", "");
		int separatorLoc = command.indexOf('=');
		String parameter = command.substring(0,separatorLoc);
		String value = command.substring(separatorLoc + 1, command.length());
		if (parameter == "")
		{
			if (value == "restart")
				restart();

		}else
		if (value == "?"){  //read parameters
			if (parameter == "myssid"){
				CONS.print(F("my SSID: "));
				CONS.println(ap_ssid);
			}
			else if(parameter == "rfpwr"){
				CONS.print(F("rf Power: "));
				CONS.println(rf_power);
			}
			else if(parameter == "apssid"){
				CONS.print(F("AP SSID: "));
				CONS.println(wifi_ssid);

			}
			else if(parameter == "appass"){
				CONS.print(F("AP PASS: "));
				CONS.println(wifi_pass);
			}
			else if(parameter == "burn"){
				CONS.println(F("Format is burn=true"));
			}
			else{
				CONS.println(F("Parameter Error"));
			}
		}
		else{ //write parameters
			if(parameter == "myssid"){
				if (value.length() < 32){
					ap_ssid = value;
					CONS.print(F("my SSID: "));
					CONS.println(ap_ssid);
					return String("mySSID updated to " + String(ap_ssid));
				}
				else{
					CONS.println(F("SSID too long"));
					return "mySSID bad format";
				}
			}
			else if(parameter == "rfpwr"){
				if ((value.toInt() < 1) || (value.toInt() > 80)){
					CONS.println("0 < rfpwr < 80");
					return ("rfPWR bad format");
				}
				else{
					rf_power = value.toInt();
					CONS.print("my rfPWR: ");
					CONS.println(rf_power);
					return String(("rfPWR updated to") + String(rf_power));
				}
			}
			else if(parameter == "apssid"){
				if (value.length() < 32){
					wifi_ssid = value;
					CONS.print("AP SSID: ");
					CONS.println(wifi_ssid);
					return String("apSSID updated to" + String(wifi_ssid));
				}
				else{
					CONS.println("SSID too long");
					return ("apSSID bad format");
				}
			}
			else if(parameter == "appass"){
				if (value.length() < 64){
					wifi_pass = value;
					CONS.print(F("AP PASS: "));
					CONS.println(wifi_pass);
					return String(("apPASS updated to ") + String(wifi_pass));
				}
				else{
					CONS.println(F("PASS too long"));
					return ("apPASS bad format");
				}
			}
			else if(parameter == "burn"){
				if (value == "true"){
					settings_save();
					CONS.println(F("Params Burned"));
					delay(100);
					pinMode(5, OUTPUT);
					digitalWrite(5, LOW);
					//system_restart();
				}
				else if (value == "default"){
					settings_default();
					CONS.println("Params Burned");
					delay(100);
					pinMode(5, OUTPUT);
					digitalWrite(5, LOW);
					//system_restart();
				}
				else if (value == "erase"){
					settings_remove();
					CONS.println(F("Params Erased"));
					delay(100);
					pinMode(5, OUTPUT);
					digitalWrite(5, LOW);
					//system_restart();
				}
				else{
					CONS.println(F("Format bad"));
				}
			}
		}
		return "";
	}

	void Axon::consoleLoop()
	{
		if (CONS.available() > 0)
		{
			CONS.setTimeout(1000);
			byte len = CONS.readBytesUntil('\n', serBuffer, 256);

			if (len >= 256){
				CONS.println("Overflow");
			}
			else{
				CONS.write(serBuffer, len);
				CONS.println();
				//null termination
				serBuffer[len] = 0;
				//string object
				String command(serBuffer);
				processCommand(command);
			}
			cursor();
		}
	}

	void Axon::cursor()
	{
		CONS.print(">");
	}

	//write Json from serial
	bool Axon::settings_jsonWrite(String jsonInput)
	{
		File settingsFile = SPIFFS.open("/settings.json", "w");
		if (!settingsFile)
		{
			CONS << F("Failed to open config file for writing ");
			return false;
		}

		settingsFile.print(jsonInput);
		settingsFile.close();
		return true;

	}
*/

	//send settings.json on serial
	void Axon::settings_output()
	{
		if (SPIFFS.begin())
		{
			LOG << F("OK") << endl;
			File settingsFile = SPIFFS.open("/settings.json", "r");
			if (!settingsFile)
			{
				LOG << F("Failed to open settings file  ");
				return;
			}

			size_t size = settingsFile.size();
			if (size > 500)
			{
				LOG << F("Settings file size is too large ");
				return;
			}

			std::unique_ptr<char[]> buf(new char[size]);
			settingsFile.readBytes(buf.get(), size);
			Serial.print(buf.get());
		}
		else
		{
			LOG << F("ERROR") << endl;
		}
	}

	Stream::Stream()
	{
	}

	bool Stream::put(String value, uint8_t qos, bool retained)
	{
		bool ret = true;

		unsigned long _s;

		this->value = value;
		_s = millis();
		LOG << F("> PUT stream ") << uid;// << F(" <- ") << value;

		ret &= axon->mqttClient.connected() &&
			     axon->mqttClient.publish(MQTT::Publish(mqtt_topic, value).set_qos(qos).set_retain(retained));

		LOG << (ret ? F(" - OK") : F(" - ERROR"));
		LOG << " " << _DEC(millis() - _s) << "ms" << endl;

		/*
		_s = millis();
		LOG << F("Putting (REST) to stream ") << uid << F(" <- ") << value;
		LOG << (put_rest(value) ? F(" - OK") : F(" - ERROR"));
		LOG << " " << _DEC(millis() - _s) << "ms" << endl;
		*/
		return ret;
	}

	bool Stream::got(const MQTT::Publish& pub)
	{
		LOG << F("> GOT stream ") << uid;
		this->value = pub.payload_string();

		if (handle)
		{
			LOG << F(" & handler invoked") << endl;
			//String topic = static_cast<String>(pub.topic());
			String message = static_cast<String>(pub.payload_string());

			handle(*this, message);
			yield();
		}
		else
			LOG << endl;
	}

	bool Stream::tail(bool full = false)
	{
		bool ret = true;

		//LOG << F("> TAIL stream ") << uid << F(" @ ") << axon->ticks << F(" - ");

		if (full)
		{
			ret &= axon->mqttClient.publish(MQTT::Publish(mqtt_topic+"/uid", uid).set_qos(1).set_retain(true));
			ret &= axon->mqttClient.publish(MQTT::Publish(mqtt_topic+"/name", name).set_qos(1).set_retain(true));
			ret &= axon->mqttClient.publish(MQTT::Publish(mqtt_topic+"/type", Helper::parse(type)).set_qos(1).set_retain(true));
			ret &= axon->mqttClient.publish(MQTT::Publish(mqtt_topic+"/acl", Helper::parse(acl)).set_qos(1).set_retain(true));
		}
		ret &= axon->mqttClient.publish(MQTT::Publish(mqtt_topic+"/tick", String(axon->ticks())).set_qos(1));

		yield();

		//LOG << (ret ? F("OK") : F("ERROR")) << endl;
		if (!ret) LOG << F("> TAIL stream ") << uid << F(" @ ") << axon->ticks() << F(" - ERROR") << endl;

		return ret;
	}
	/*
	bool Stream::put_rest(String value)
	{
		//LOG << F("H{ ") << ESP.getFreeHeap() << F(" B") << endl;

		//LOG << F("   Connecting           : ");
		if (axon->wifiSecureClient.connect(axon->thingy_server.c_str(), axon->thingy_port))
		{
			//LOG << F("OK") << endl;
		}
		else
		{
			//LOG << F("ERROR") << endl;
			return false;
		}
		yield();

		//LOG << F("   Checking certificate : ");
		if (axon->wifiSecureClient.verify(axon->thingy_finger.c_str(), axon->thingy_server.c_str()))
		{
			//LOG << F("Verified") << endl;
		} else
		{
			//LOG << F("Unverified") << endl;
		}
		yield();

		String url = "/streams/" + uid;
		//LOG << F("   API identifier       : ") << url << endl;

		////LOG << F("   Requesting           : ");
		String data = "{\"value\":\""+value+"\"}";

		String request =
			"PUT " + url + " HTTP/1.1\r\n" +
			"Host: " + axon->thingy_server + "\r\n" +
			"User-Agent: Axon." + axon->uid + "\r\n" +
			"Authorization: Basic " + base64::encode(axon->sid+":") +"\r\n" +
			"Content-Type: application/json" +"\r\n" +
			"Content-Length: "+data.length()+"\r\n" +
			"Connection: close\r\n\r\n" +
			data;


		//LOG << F("===== request =====") << endl << request << endl << F("===================") << endl << endl;

		if (axon->wifiSecureClient.print(request))
		{
			//LOG << F("OK") << endl;
		}
		else
		{
			//LOG << F("ERROR") << endl;
			return false;
		}

		//LOG << F("   Receiving            : ") << endl;
		//LOG << F("===== headers =====") << endl;
		while (axon->wifiSecureClient.connected())
		{
			while (!axon->wifiSecureClient.available())
				yield();

			String line = axon->wifiSecureClient.readStringUntil('\n');

			if (line == "\r")
			{
				break;
			}
			else
			{
				//LOG << line << endl;
			}
		}
		//LOG << F("===================") << endl << endl;

		while (!axon->wifiSecureClient.available())
				yield();

		String response = axon->wifiSecureClient.readStringUntil('\n');

		yield();

		axon->wifiSecureClient.stop();

		//LOG << F("===== response =====") << endl << response << endl << F("====================") << endl << endl;

		//LOG << F("   Parsing response     : ");
		StaticJsonBuffer<1000> jsonBuffer;
		JsonObject& root = jsonBuffer.parseObject(response);
		if (root.success())
		{
			//LOG << F("OK") << endl;
		}
		else
		{
			//LOG << F("ERROR") << endl;
			return false;
		}

		//LOG << F("H} ") << ESP.getFreeHeap() << endl;
		//LOG << F("UiD = ") << (//CONSt char*)(root["UiD"]) << endl;

		return true;

	}
	*/

}
