/*
	Thingy library

	This library that enables connectivity to Thingy.IO system.

	Copyright (c) 2017 Thingy.IO
*/

#define DEBUG true
#define LOG if(DEBUG)Serial

#define CONS Serial

#ifndef thingy_h
#define thingy_h

#include <functional>
#include <Arduino.h>

#include <SPI.h>

#include <ArduinoJson.h>
#include <FS.h>

#include <Hash.h>
#include <base64.h>

#include <ESP8266WiFi.h>
#include <PubSubClient.h>

#include <Streaming.h>

#include <ESP8266mDNS.h>
#include <WiFiUdp.h>
#include <ArduinoOTA.h>

#include <DNSServer.h>
#include <ESP8266WebServer.h>
#include <WiFiManager.h>

namespace Thingy
{
	class Brain;
	class Axon;
	class Stream;

	enum StreamType
	{
		Dendrite, // goes to axon, readable by device, writable by outside
		Terminal  // goes from axon, writtable by device, readable by outside
	};

	enum StreamAccessLevel
	{
		System,
		Private,
		Public
	};

	class Stream
	{
		friend Axon;

		private:
			typedef std::function<void(const Thingy::Stream&, const String&)> StreamHandlerType;
			StreamHandlerType handle = NULL;

			String mqtt_topic;

			bool put_mqtt(String value);
			bool put_rest(String value);

		protected:
			bool got(const MQTT::Publish& pub);
			bool tail(bool full);

		public:
			Axon* axon;

			String uid;
			String name;
			StreamType type;
			StreamAccessLevel acl;

			String value;

			Stream();

			bool put(String value, uint8_t qos = 0, bool retained = false);

			void setHandle(StreamHandlerType h) { handle = h; }
			void resetHandle(StreamHandlerType h) { handle = NULL; }
	};

	class Axon
	{
		friend Stream;

		private:
			typedef std::function<void(const Thingy::Axon&, const String&, const String&, const MQTT::Publish&)> AxonHandlerType;
			AxonHandlerType handle = NULL;

			// hardware
			String chipId;

			// runtime
			long drops = 0;
			unsigned long _ticks = 0;
			bool   hooked = false;

			//console stuff
			//char serBuffer[32];
			//String processCommand(String command);

			//
			String settings;
			unsigned long settings_timestamp=0;
			//

			String brain;

			String wifi_ssid;
			String wifi_pass;

			String ap_ssid;
			String ap_pass;

			Stream* streams;
			byte streams_len=0;

		protected:
			WiFiClient wifiClient;
			PubSubClient mqttClient;
			WiFiClientSecure wifiSecureClient;
			//WiFiClientSecure wifiSecureClientMQTT;

			bool head();
			bool tail(bool full);
			void got(const MQTT::Publish& pub);

		public:
			String mqtt_server;
			int    mqtt_port;

			int rf_power;

			Thingy::Stream& operator[](String nameOrUid);

			String thingy_server;
			int    thingy_port;
			String thingy_finger;

			String uid;
			String sid;
			String eid;
			String name;

			Axon();
			void setup();

			Axon& set_uid(String uid);
			Axon& set_sid(String sid);
			Axon& set_wifi_ssid(String ssid);
			Axon& set_wifi_pass(String pass);

			bool begin();

			void loop();
			void restart();

			bool connect();

			long unsigned int ticks() { return _ticks;}
			bool connected();

			// wifi
			String wifi_ip;

			void wifi_setup(String ssid, String password);
			bool wifi_connect();
			bool wifi_connected();

			bool thingy_connected() { return hooked; }

			// mqtt
			String mqtt_topic;
			String mqtt_client_id;

			bool mqtt_loop();
			bool mqtt_connect();
			void mqtt_disconnect();
			bool mqtt_connected();

			// settings
			bool settings_load();
			bool settings_save();
			bool settings_default();
			bool settings_remove();
			void settings_web_portal();


			void settings_output();
			bool settings_jsonWrite(String jsonInput);

			//***//

			//console stuff
			void consoleLoop();
			void cursor();

			// listen AP
			void scan_wifi_networks(String& table);

			// notifications
			void setHandle(AxonHandlerType h) { handle = h; }
			void resetHandle(AxonHandlerType h) { handle = NULL; }

			bool publish(MQTT::Publish &pub);

			//wifi setup mode
			void setupLoop(void);
			int setupCounter = 0;
			bool setupMode = false;
	};

	class Brain
	{
		private:

		public:
			String uid;
			String name;
			String description;

			Brain();
	};

	class Helper
	{
		public:
			static String uptime_formatted()
			{
				char temp[10];
				int sec = millis() / 1000;
				int min = sec / 60;
				int hr = min / 60;
				snprintf ( temp, 400, "%02d:%02d:%02d", hr, min % 60, sec % 60 );
				return String(temp);
			}

			static String parse(Thingy::StreamType stream_type)
			{
				if (stream_type == StreamType::Dendrite)
					return "dendrite";
				if (stream_type == StreamType::Terminal)
					return "terminal";
			}

			static String parse(Thingy::StreamAccessLevel stream_access_level)
			{
				if (stream_access_level == StreamAccessLevel::System)
					return "system";
				if (stream_access_level == StreamAccessLevel::Private)
					return "private";
				if (stream_access_level == StreamAccessLevel::Public)
					return "public";
			}

	};
}

#endif
